<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\CartItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Notifications\OrderPlacedNotification;

class OrderController extends Controller
{
    public function index()
    {
        $orders = Order::where('user_id', Auth::id())->with('orderItems.product')->get();
        return view('orders.index', compact('orders'));
    }

    public function store()
    {
        $cartItems = CartItem::with('product')->where('user_id', Auth::id())->get();

        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')->with('error', 'Your cart is empty.');
        }

        DB::transaction(function () use ($cartItems) {
            $order = Order::create([
                'user_id' => Auth::id(),
                'total_price' => $cartItems->sum(fn($item) => $item->product->price * $item->quantity),
                'status' => 'pending',
            ]);

            foreach ($cartItems as $cartItem) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $cartItem->product_id,
                    'quantity' => $cartItem->quantity,
                    'price' => $cartItem->product->price,
                ]);
            }

            CartItem::where('user_id', Auth::id())->delete();

            // Send order confirmation notification
            $order->user->notify(new OrderPlacedNotification($order));
        });

        return redirect()->route('orders.index')->with('status', 'Order placed successfully.');
    }

    public function show(Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }

        $order->load('orderItems.product');
        return view('orders.show', compact('order'));
    }

    public function updateStatus(Request $request, Order $order)
    {
        $request->validate([
            'status' => 'required|string|in:shipped,delivered,returned,refunded',
        ]);

        $user = Auth::user();

        if ($user->role === 'seller' && $order->status !== 'pending') {
            return response()->json(['error' => 'Only pending orders can be updated by sellers.'], 403);
        }

        if ($user->role === 'buyer' && $request->status !== 'delivered') {
            return response()->json(['error' => 'Buyers can only mark orders as delivered.'], 403);
        }

        if ($user->role === 'admin' || $user->role === 'support') {
            // Admins and support can update to any status
            $order->update(['status' => $request->status]);
            return response()->json(['success' => 'Order status updated successfully.']);
        }

        if ($user->role === 'seller' && $request->status === 'shipped') {
            $order->update(['status' => 'shipped']);
            return response()->json(['success' => 'Order marked as shipped.']);
        }

        if ($user->role === 'buyer' && $request->status === 'delivered') {
            $order->update(['status' => 'delivered']);
            return response()->json(['success' => 'Order marked as delivered.']);
        }

        return response()->json(['error' => 'Unauthorized action.'], 403);
    }
}